/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.spi.impl;

import io.moquette.persistence.PersistentSession;
import io.moquette.spi.ClientSession;
import io.moquette.spi.DurableSession;
import io.moquette.spi.ISessionsStore;
import io.moquette.spi.ISubscriptionsStore;
import io.moquette.spi.TransientSession;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionsRepository {
    private static final Logger LOG = LoggerFactory.getLogger(SessionsRepository.class);
    private final ISessionsStore sessions;
    private ISubscriptionsStore subscriptionsStore;
    private ScheduledExecutorService scheduler;
    private final Map<String, ClientSession> sessionsCache = new ConcurrentHashMap<String, ClientSession>();

    public SessionsRepository(ISessionsStore sessionsStore, ScheduledExecutorService scheduler) {
        this.sessions = sessionsStore;
        this.subscriptionsStore = sessionsStore.subscriptionStore();
        this.scheduler = scheduler;
    }

    public void init() {
        SessionCleanerTask cleanerTask = new SessionCleanerTask();
        this.scheduler.schedule(cleanerTask, 1L, TimeUnit.HOURS);
    }

    public ClientSession sessionForClient(String clientID) {
        LOG.debug("Retrieving session. CId={}", (Object)clientID);
        if (this.sessionsCache.containsKey(clientID)) {
            return this.sessionsCache.get(clientID);
        }
        if (!this.sessions.contains(clientID)) {
            LOG.warn("Session does not exist. CId={}", (Object)clientID);
            return null;
        }
        PersistentSession storedSession = this.sessions.loadSessionByKey(clientID);
        return this.newClientSessionAndCacheIt(storedSession.clientID, storedSession.cleanSession);
    }

    private ClientSession newClientSessionAndCacheIt(String clientID, boolean cleanSession) {
        ClientSession session;
        if (cleanSession) {
            session = new TransientSession(clientID);
        } else {
            DurableSession durableSession = new DurableSession(clientID, this.sessions, this.subscriptionsStore);
            durableSession.reloadAllSubscriptionsFromStore();
            session = durableSession;
        }
        this.sessionsCache.put(clientID, session);
        return session;
    }

    public ClientSession createNewSession(String clientID, boolean cleanSession) {
        if (this.sessions.contains(clientID)) {
            LOG.error("Unable to create a new session: the client ID is already in use. CId={}, cleanSession={}", (Object)clientID, (Object)cleanSession);
            throw new IllegalArgumentException("Can't create a session with the ID of an already existing" + clientID);
        }
        LOG.debug("Creating new session. CId={}, cleanSession={}", (Object)clientID, (Object)cleanSession);
        if (!cleanSession) {
            this.sessions.createNewDurableSession(clientID);
        }
        return this.newClientSessionAndCacheIt(clientID, cleanSession);
    }

    public Collection<ClientSession> getAllSessions() {
        ArrayList<ClientSession> result = new ArrayList<ClientSession>();
        for (PersistentSession persistentSession : this.sessions.listAllSessions()) {
            result.add(this.sessionForClient(persistentSession.clientID));
        }
        return result;
    }

    private void updateCleanStatus(String clientID, boolean cleanSession) {
        LOG.info("Updating cleanSession flag. CId={}, cleanSession={}", (Object)clientID, (Object)cleanSession);
        this.sessions.updateCleanStatus(clientID, cleanSession);
    }

    ClientSession createOrLoadClientSession(String clientId, boolean cleanSession) {
        ClientSession clientSession = this.sessionForClient(clientId);
        if (clientSession == null) {
            clientSession = this.createNewSession(clientId, cleanSession);
        } else if (!clientSession.isCleanSession() && cleanSession) {
            this.sessions.removeDurableSession(clientId);
            clientSession = this.createNewSession(clientId, true);
        }
        if (cleanSession) {
            LOG.info("Cleaning session. CId={}", (Object)clientId);
            clientSession.cleanSession();
        }
        return clientSession;
    }

    public void disconnect(String clientId) {
        LOG.debug("Removing session from repository's cache");
        ClientSession clientSession = this.sessionForClient(clientId);
        if (clientSession == null) {
            return;
        }
        this.sessionsCache.remove(clientId);
        this.sessions.trackSessionClose(LocalDateTime.now(), clientId);
    }

    private void wipeExpiredSessions() {
        LocalDateTime pin = LocalDateTime.now().minus(6L, ChronoUnit.DAYS);
        Set<String> expiredSessionsIds = this.sessions.sessionOlderThan(pin);
        for (String expiredSession : expiredSessionsIds) {
            this.sessions.removeDurableSession(expiredSession);
            this.subscriptionsStore.wipeSubscriptions(expiredSession);
        }
    }

    private class SessionCleanerTask
    implements Runnable {
        private SessionCleanerTask() {
        }

        @Override
        public void run() {
            SessionsRepository.this.wipeExpiredSessions();
        }
    }
}

